// otrn_lib.js

import { v4 as uuidv4 } from 'uuid'; // Requires 'npm install uuid'
import fs from 'fs';
import path from 'path';

/**
 * Helper function to convert a class instance to a plain object (similar to asdict).
 * It recursively cleans up properties with a value of 'undefined' or 'null' 
 * but keeps required empty arrays.
 * @param {Object} instance - The class instance to convert.
 * @returns {Object} A plain JavaScript object.
 */
function cleanToDict(instance) {
    const data = JSON.parse(JSON.stringify(instance)); // Simple deep copy of serializable props

    const requiredEmptyLists = ['notes', 'replies', 'tags', 'files']; // 'files' is added here for OTRNDocument

    function clean(d) {
        if (Array.isArray(d)) {
            return d.map(clean);
        } else if (typeof d === 'object' && d !== null) {
            const cleaned = {};
            for (const [k, v] of Object.entries(d)) {
                // If value is null/undefined, skip it unless it's a required empty array
                if (v !== null && v !== undefined) {
                    if (Array.isArray(v) && v.length === 0 && !requiredEmptyLists.includes(k)) {
                        // Skip optional empty arrays
                        continue;
                    }
                    cleaned[k] = clean(v);
                }
            }
            return cleaned;
        }
        return d;
    }
    
    // The top-level 'files' array is optional and should be removed if empty
    let cleanedData = clean(data);
    if (Array.isArray(cleanedData.files) && cleanedData.files.length === 0) {
        delete cleanedData.files;
    }

    return cleanedData;
}


// --- 1. Nested Objects ---

/**
 * Represents a single reply object within a Note (Section 8).
 */
export class Reply {
    /**
     * @param {string} comment - Required.
     * @param {string} [commenter=undefined]
     * @param {string} [timestamp=undefined] - ISO 8601
     * @param {string} [replyId=undefined]
     */
    constructor({ comment, commenter, timestamp, replyId }) {
        this.comment = comment; // Required
        this.commenter = commenter;
        this.timestamp = timestamp;
        this.replyId = replyId;
    }
}

/**
 * Represents a single tag object within a Note (Section 9).
 */
export class Tag {
    /**
     * @param {string} name - Required.
     * @param {string} [tagId=undefined]
     * @param {string} [group=undefined]
     */
    constructor({ name, tagId, group }) {
        this.name = name; // Required
        this.tagId = tagId;
        this.group = group;
    }
}

/**
 * Represents a single timecode-related note, including range and nested objects (Section 7).
 */
export class Note {
    /**
     * @param {object} params
     * @param {number} params.time - Required. The exact time in seconds.
     * @param {string} params.comment - Required. The main body of the note.
     * @param {string} [params.timecode=undefined] - HH:MM:SS:FF or HH:MM:SS;FF
     * @param {number} [params.frame=undefined]
     * @param {boolean} [params.range=false]
     * @param {number} [params.timeOut=undefined]
     * @param {string} [params.timecodeOut=undefined]
     * @param {number} [params.frameOut=undefined]
     * @param {string} [params.name=undefined]
     * @param {string} [params.commenter=undefined]
     * @param {string} [params.color=undefined]
     * @param {string} [params.colorHex=undefined]
     * @param {string} [params.category=undefined]
     * @param {string} [params.track=undefined]
     * @param {boolean} [params.complete=false]
     * @param {string} [params.timestamp=undefined] - ISO 8601 creation/update time
     * @param {string} [params.noteId=undefined] - Unique ID
     * @param {Array<Reply>} [params.replies=[]]
     * @param {Array<Tag>} [params.tags=[]]
     */
    constructor({
        time, comment, timecode, frame, range = false, timeOut, timecodeOut, frameOut,
        name, commenter, color, colorHex, category, track, complete = false, timestamp, 
        noteId = uuidv4(), replies = [], tags = [] 
    }) {
        // Required
        this.time = time;
        this.comment = comment;

        // Optional Timecode
        this.timecode = timecode;
        this.frame = frame;

        // Optional Range Marker
        this.range = range;
        this.timeOut = timeOut;
        this.timecodeOut = timecodeOut;
        this.frameOut = frameOut;

        // Optional Note Metadata
        this.name = name;
        this.commenter = commenter;
        this.color = color;
        this.colorHex = colorHex;
        this.category = category;
        this.track = track;
        this.complete = complete;
        this.timestamp = timestamp;
        this.noteId = noteId;

        // Optional Nested Arrays
        this.replies = replies;
        this.tags = tags;
    }
}


// --- 2. Top-Level Objects ---

/**
 * Represents the top-level 'metadata' object using CamelCase (Section 5).
 */
export class Metadata {
    /**
     * @param {object} [params={}]
     * @param {string} [params.project=undefined]
     * @param {string} [params.fileName=undefined]
     * @param {string} [params.fileUrl=undefined]
     * @param {string} [params.software=undefined]
     * @param {number} [params.otrnVersion=1] - Default version 1
     * @param {string} [params.otrnInfo="..."]
     * @param {string} [params.timestamp=ISO 8601] - ISO 8601 with Z
     */
    constructor({
        project, fileName, fileUrl, software, otrnVersion = 1,
        otrnInfo = "This is an OTRN (Open Timecode-Related Notes) notes file. Learn more about the specification on https://otrn.editingtools.io",
        timestamp = new Date().toISOString().replace(/\.\d{3}Z$/, 'Z') // ISO 8601 with Z
    } = {}) {
        this.project = project;
        this.fileName = fileName;
        this.fileUrl = fileUrl;
        this.software = software;
        this.otrnVersion = otrnVersion;
        this.otrnInfo = otrnInfo;
        this.timestamp = timestamp;
    }
}

/**
 * Base class for Sequence and File objects, holding common timecode attributes (Sections 6 & 10).
 */
class SequenceFileBase {
    /**
     * @param {object} params
     * @param {string} [params.name=undefined]
     * @param {number} [params.frameRate=undefined]
     * @param {boolean} [params.dropFrame=false]
     * @param {number} [params.startTime=0.0]
     * @param {string} [params.startTimecode="00:00:00:00"]
     * @param {number} [params.startFrame=0]
     * @param {Array<Note>} [params.notes=[]] - Required array
     */
    constructor({
        name, frameRate, dropFrame = false, startTime = 0.0,
        startTimecode = "00:00:00:00", startFrame = 0, notes = []
    }) {
        this.name = name;
        this.frameRate = frameRate;
        this.dropFrame = dropFrame;
        this.startTime = startTime;
        this.startTimecode = startTimecode;
        this.startFrame = startFrame;
        this.notes = notes;
    }
}

/**
 * Represents the top-level 'sequence' object (Section 6).
 */
export class Sequence extends SequenceFileBase {
    // Inherits all fields from SequenceFileBase
    constructor(params = {}) {
        super(params);
    }
}

/**
 * Represents a single object in the top-level 'files' array for clip notes (Section 10).
 */
export class File extends SequenceFileBase {
    /**
     * @param {object} params
     * @param {string} [params.name=undefined]
     * @param {number} [params.frameRate=undefined]
     * @param {boolean} [params.dropFrame=false]
     * @param {number} [params.startTime=0.0]
     * @param {string} [params.startTimecode="00:00:00:00"]
     * @param {number} [params.startFrame=0]
     * @param {Array<Note>} [params.notes=[]]
     * @param {string} [params.fileName=""] - Required
     * @param {string} [params.filePath=""] - Required
     * @param {string} [params.clipName=undefined]
     */
    constructor({ fileName = "", filePath = "", clipName, ...baseParams }) {
        super(baseParams);
        this.fileName = fileName; // Required
        this.filePath = filePath; // Required
        this.clipName = clipName;
    }
}

/**
 * The root structure for an Open Timecode-Related Notes file (Section 4).
 * Contains metadata, sequence (required), and files (optional).
 */
export class OTRNDocument {
    /**
     * @param {object} [params={}]
     * @param {Metadata} [params.metadata=new Metadata()]
     * @param {Sequence} [params.sequence=new Sequence()]
     * @param {Array<File>} [params.files=[]] - Optional array
     */
    constructor({ metadata = new Metadata(), sequence = new Sequence(), files = [] } = {}) {
        this.metadata = metadata;
        this.sequence = sequence;
        this.files = files;
    }

    /**
     * Converts the OTRNDocument and all nested classes to a clean dictionary/object.
     * @returns {Object}
     */
    toDict() {
        return cleanToDict(this);
    }
}

// --- Library Functions ---

/**
 * Helper to instantiate a list of dataclasses from a raw list of objects.
 * @param {Class} cls - The class constructor (e.g., Note, Reply, Tag).
 * @param {Array<Object>} dataList - The raw list of objects.
 * @returns {Array<Object>} An array of class instances.
 */
function _instantiateList(cls, dataList) {
    if (!Array.isArray(dataList)) return [];
    return dataList.map(item => new cls(item));
}

/**
 * Helper function to process and instantiate a list of Note objects, 
 * including their nested replies and tags.
 * @param {Array<Object>} notesData - Raw list of note objects.
 * @returns {Array<Note>}
 */
function processNotes(notesData) {
    const notes = [];
    if (!Array.isArray(notesData)) return [];
    for (const noteData of notesData) {
        // Instantiate nested arrays first
        const replies = _instantiateList(Reply, noteData.replies);
        const tags = _instantiateList(Tag, noteData.tags);
        
        // Remove keys from raw data to avoid conflicts with required list structure
        delete noteData.replies;
        delete noteData.tags; 
        
        notes.push(new Note({ ...noteData, replies, tags }));
    }
    return notes;
}

/**
 * Reads an OTRN JSON file and returns an OTRNDocument object.
 * @param {string} filepath - Path to the OTRN file.
 * @returns {OTRNDocument}
 */
export function readOTRN(filepath) {
    console.log(`Reading OTRN file from: ${filepath}`);
    const data = JSON.parse(fs.readFileSync(path.resolve(filepath), 'utf8'));

    // 1. Process Metadata
    const metadata = new Metadata(data.metadata || {});

    // 2. Process Sequence
    const sequenceData = data.sequence || {};
    sequenceData.notes = processNotes(sequenceData.notes);
    const sequence = new Sequence(sequenceData);

    // 3. Process Files (optional)
    const files = [];
    const filesData = data.files || [];
    for (const fileData of filesData) {
        fileData.notes = processNotes(fileData.notes);
        files.push(new File(fileData));
    }

    // 4. Construct the OTRNDocument
    const document = new OTRNDocument({
        metadata: metadata,
        sequence: sequence,
        files: files
    });

    console.log("Successfully loaded OTRN document.");
    return document;
}

/**
 * Writes an OTRNDocument object to a JSON file.
 * @param {OTRNDocument} document - The OTRNDocument object.
 * @param {string} filepath - Path to the output JSON file.
 */
export function writeOTRN(document, filepath) {
    console.log(`Writing OTRN document to: ${filepath}`);

    // Convert the document object to a clean dictionary
    const data = document.toDict();

    fs.writeFileSync(path.resolve(filepath), JSON.stringify(data, null, 4), 'utf8');

    console.log("Successfully wrote OTRN document.");
}