// reader_example.js

import { readOTRN } from './otrn_lib.js';
import fs from 'fs';
import path from 'path';

const OTRN_INPUT_FILE = "example_data_to_read.otrn";
const OTRN_ARRAY_FILE = "array_output.otrn";

/**
 * Reads the OTRN document created by writer_example.js and prints its contents.
 */
function mainReadExample() {

    if (!fs.existsSync(OTRN_INPUT_FILE)) {
        console.log(`Error: Input file '${OTRN_INPUT_FILE}' not found.`);
        console.log("Please run 'writer_example.js' first to generate the necessary file.");
        return;
    }

    console.log("--- Starting OTRN Reader Example ---");
    const readDoc = readOTRN(OTRN_INPUT_FILE);
    
    console.log(`\nDocument Project: ${readDoc.metadata.project}`);

    // --- Listing Sequence Notes ---
    console.log("\n" + "=".repeat(40));
    console.log("      Listing Sequence Notes");
    console.log("=".repeat(40));
    
    if (readDoc.sequence.notes && readDoc.sequence.notes.length > 0) {
        readDoc.sequence.notes.forEach((note, i) => {
            console.log(`Note #${i+1} [Sequence]`);
            console.log(`  Time: ${note.time}s (${note.timecode || 'N/A'})`);
            console.log(`  Comment: ${note.comment}`);
            console.log(`  Name: ${note.name}`);
            console.log(`  Commenter: ${note.commenter}`);
            console.log(`  Status: ${note.complete ? 'Complete' : 'Pending'}`);
            
            if (note.range) {
                console.log(`  Range End: ${note.timeOut}s (${note.timecodeOut || 'N/A'})`);
            }
            if (note.tags && note.tags.length > 0) {
                const tagNames = note.tags.map(t => t.name).join(', ');
                console.log(`  Tags: ${tagNames}`);
            }
            if (note.replies && note.replies.length > 0) {
                 console.log(`  Replies: ${note.replies.length} total`);
            }
            console.log("-".repeat(40));
        });
    } else {
        console.log("No sequence notes found.");
    }

    // --- Listing File (Clip) Notes ---
    console.log("\n" + "=".repeat(40));
    console.log("      Listing File (Clip) Notes");
    console.log("=".repeat(40));
    
    if (readDoc.files && readDoc.files.length > 0) {
        readDoc.files.forEach(fileObj => {
            console.log(`File Object: ${fileObj.fileName} (Clip: ${fileObj.clipName || 'N/A'})`);
            if (fileObj.notes && fileObj.notes.length > 0) {
                fileObj.notes.forEach((note, i) => {
                    console.log(`  Note #${i+1} [File]`);
                    console.log(`    Time: ${note.time}s (${note.timecode || 'N/A'})`);
                    console.log(`    Comment: ${note.comment}`);
                    console.log(`    Name: ${note.name}`);
                    console.log(`    Commenter: ${note.commenter}`);
                });
                console.log("-".repeat(40));
            } else {
                console.log("  No notes found for this file.");
            }
        });
    } else {
        console.log("No file objects with notes found.");
    }
}

/**
 * Reads an OTRN document and converts its sequence notes into a plain JavaScript array.
 * @param {string} filepath - Path to the OTRN file.
 * @returns {Array<Object>} An array of plain note objects.
 */
function otrnToArrayExample(filepath) {
    if (!fs.existsSync(filepath)) {
        console.log(`\nError: Input file '${filepath}' not found.`);
        console.log("Skipping array read example.");
        return [];
    }

    console.log("\n" + "=".repeat(40));
    console.log("  Starting OTRN to Array Example");
    console.log("=".repeat(40));
    
    const arrayDoc = readOTRN(filepath);

    if (arrayDoc.sequence.notes && arrayDoc.sequence.notes.length > 0) {
        // Use map on the notes array and Note.toDict (via cleanToDict) to get a clean object for each
        const noteArray = arrayDoc.sequence.notes.map(note => note); 
        console.log(`\nSuccessfully converted ${noteArray.length} sequence notes to a JavaScript Array.`);
        
        // Print the first element for demonstration
        console.log("First element of the resulting array:");
        console.log(JSON.stringify(noteArray[0], null, 2));
        
        return noteArray;
    } else {
        console.log("No sequence notes found in the array example file.");
        return [];
    }
}


if (import.meta.url === path.fileURLToPath(import.meta.url)) {
    mainReadExample();
    otrnToArrayExample(OTRN_ARRAY_FILE);
}