// writer_example.js

import { OTRNDocument, Metadata, Sequence, Note, File, Reply, Tag, writeOTRN } from './otrn_lib.js';

// Define the file name that the writer will output
const OTRN_OUTPUT_FILE = "example_data_to_read.otrn";

/**
 * Creates a fully populated OTRNDocument object ready for serialization.
 * @returns {OTRNDocument}
 */
function createExampleDocument() {
    
    // 1a. Define Metadata
    const metadataObj = new Metadata({
        project: "Quantum Mechanics Documentary (Writer)",
        fileName: "Episode_Pilot_V03.mov",
        fileUrl: "ftp://server.com/docs/Episode_Pilot_V03.mov",
        software: "OTRN JavaScript Writer"
    });

    // 1b. Define nested Reply and Tag
    const replyObj = new Reply({
        commenter: "Sound Mixer",
        comment: "This audio issue is fixed in the latest mix (V05).",
        timestamp: new Date().toISOString().replace(/\.\d{3}Z$/, 'Z'), // ISO 8601 with Z
        replyId: "rep-12345"
    });
    
    const tagObj = new Tag({
        name: "Archive",
        tagId: "archive_tag",
        group: "Footage Management"
    });

    // 1c. Define Notes for the Sequence
    const note1 = new Note({
        time: 300.0,
        comment: "Great visual transition here. Flag for the sizzle reel.",
        commenter: "Director",
        color: "lime",
        complete: true,
        noteId: "seq-n1"
    });
    
    const note2 = new Note({
        time: 120.5,
        timecode: "01:02:00:12",
        comment: "The host's microphone pops here. Needs de-essing or replacement.",
        commenter: "Audio Editor",
        color: "orange",
        category: "Audio",
        track: "A1",
        complete: false,
        range: true,
        timeOut: 122.8,
        timecodeOut: "01:02:02:07",
        noteId: "seq-n2",
        replies: [replyObj],
        tags: [tagObj]
    });

    // Create the Sequence object
    const sequenceObj = new Sequence({
        name: "Final Edit Timeline",
        frameRate: 24.0,
        dropFrame: false,
        startTimecode: "01:00:00:00",
        notes: [note1, note2]
    });
    
    
    // 2. Create notes for files (clip notes)
    const fileNote = new Note({
        time: 5.0,
        timecode: "14:24:15:00",
        comment: "The interviewer is visibly shifting in their seat. Trim this for the final cut.",
        commenter: "Transcriptionist"
    });
    
    const fileObj = new File({
        fileName: "INTERVIEW_S01_T04.mov",
        filePath: "/path/to/rushes/INTERVIEW_S01_T04.mov",
        clipName: "Interview Subject A - Take 4",
        frameRate: 25.0,
        startTimecode: "14:24:10:00",
        notes: [fileNote]
    });


    // 3. Create the full OTRNDocument
    const newDoc = new OTRNDocument({
        metadata: metadataObj,
        sequence: sequenceObj,
        files: [fileObj]
    });
    
    return newDoc;
}

/**
 * Creates an OTRN document from a plain JavaScript array of notes.
 * @param {Array<Object>} notesArray - Array of note objects.
 * @returns {OTRNDocument}
 */
function createDocumentFromArray(notesArray) {
    const noteInstances = notesArray.map(n => {
        // Ensure nested objects (replies/tags) are instantiated if present
        const replies = (n.replies || []).map(r => new Reply(r));
        const tags = (n.tags || []).map(t => new Tag(t));
        return new Note({ ...n, replies, tags });
    });

    const metadataObj = new Metadata({
        project: "Array Import Example",
        software: "OTRN JS Array Writer"
    });
    
    const sequenceObj = new Sequence({
        name: "Imported Notes Timeline",
        notes: noteInstances
    });
    
    return new OTRNDocument({
        metadata: metadataObj,
        sequence: sequenceObj
    });
}

function mainWriteExample() {
    console.log("--- Starting OTRN Writer Example ---");
    const newDoc = createExampleDocument();
    
    console.log(`Document created for project: ${newDoc.metadata.project}`);
    console.log(`Writing document to: ${OTRN_OUTPUT_FILE}`);

    // Call the write function from the library
    writeOTRN(newDoc, OTRN_OUTPUT_FILE);
    
    console.log(`\nWriter example finished. A file named '${OTRN_OUTPUT_FILE}' has been created.`);

    // --- Array to OTRN Example ---
    console("\n" + "=" * 40);
    console.log("   Starting Array to OTRN Example");
    console.log("=" * 40);

    const notesToExport = [
        { time: 10.5, comment: "First array note.", commenter: "Array User" },
        { time: 65.0, comment: "Second note with a tag.", tags: [{ name: "Review" }] }
    ];

    const arrayDoc = createDocumentFromArray(notesToExport);
    const arrayOutputFile = "array_output.otrn";
    
    console.log(`Writing array document to: ${arrayOutputFile}`);
    writeOTRN(arrayDoc, arrayOutputFile);
    console.log(`\nArray example finished. A file named '${arrayOutputFile}' has been created.`);
}

if (import.meta.url === path.fileURLToPath(import.meta.url)) {
    mainWriteExample();
}