/**
 * otrn_lib_simple.js - Simplified OTRN Data Model Classes
 * This version contains only the OTRN data model (classes) and the serialization 
 * logic (toDict) without any external dependencies (no UUID, no fs/path).
 */

/**
 * Helper function to convert a class instance to a plain object.
 * It recursively removes properties that are undefined, null, or optional empty arrays 
 * (to keep the JSON clean and minimal according to the OTRN spec).
 * @param {Object} instance - The class instance to convert.
 * @returns {Object} A clean plain JavaScript object.
 */
function cleanToDict(instance) {
    // Required list fields that must be kept, even if empty, if their parent object is instantiated.
    const requiredEmptyLists = ['notes', 'replies', 'tags', 'files']; 

    function clean(d) {
        if (Array.isArray(d)) {
            return d.map(clean);
        } else if (typeof d === 'object' && d !== null) {
            const cleaned = {};
            for (const [k, v] of Object.entries(d)) {
                // Skip null/undefined values
                if (v === null || v === undefined) {
                    continue;
                }

                // If it's an empty array, skip it unless it's explicitly required
                if (Array.isArray(v) && v.length === 0) {
                    if (requiredEmptyLists.includes(k)) {
                         cleaned[k] = []; // Keep the required empty list
                    }
                    continue; // Skip optional empty arrays
                }
                
                // Keep primitive values and recursively clean objects/non-empty arrays
                cleaned[k] = clean(v);
            }
            return cleaned;
        }
        return d;
    }
    
    return clean(instance);
}


// --- Nested Objects ---

export class Reply {
    constructor({ comment, commenter, timestamp, replyId }) {
        this.comment = comment; // Required
        this.commenter = commenter;
        this.timestamp = timestamp;
        this.replyId = replyId;
    }
}

export class Tag {
    constructor({ name, tagId, group }) {
        this.name = name; // Required
        this.tagId = tagId;
        this.group = group;
    }
}

export class Note {
    constructor({
        time, comment, timecode, frame, range = false, timeOut, timecodeOut, frameOut,
        name, commenter, color, colorHex, category, track, complete = false, timestamp, 
        noteId, replies = [], tags = [] 
    }) {
        // Required
        this.time = parseFloat(time); // Ensure time is a float
        this.comment = comment;

        // Optional Timecode
        this.timecode = timecode;
        this.frame = frame;

        // Optional Range Marker
        this.range = range;
        this.timeOut = timeOut;
        this.timecodeOut = timecodeOut;
        this.frameOut = frameOut;

        // Optional Note Metadata
        this.name = name;
        this.commenter = commenter;
        this.color = color;
        this.colorHex = colorHex;
        this.category = category;
        this.track = track;
        this.complete = complete;
        this.timestamp = timestamp || new Date().toISOString();
        this.noteId = noteId; // No auto-generation
        
        // Optional Nested Arrays (always initialized as empty for serialization)
        this.replies = replies; 
        this.tags = tags;
    }
}


// --- Top-Level Objects ---

export class Metadata {
    constructor({
        project, fileName, fileUrl, software, otrnVersion = 1,
        otrnInfo = "This is an OTRN (Open Timecode-Related Notes) notes file. Learn more about the specification on https://otrn.editingtools.io",
        timestamp = new Date().toISOString().replace(/\.\d{3}Z$/, 'Z') // ISO 8601 with Z
    } = {}) {
        this.project = project || "Web Browser Notes";
        this.fileName = fileName;
        this.fileUrl = fileUrl;
        this.software = software || "Simple OTRN Web Creator";
        this.otrnVersion = otrnVersion;
        this.otrnInfo = otrnInfo;
        this.timestamp = timestamp;
    }
}

class SequenceFileBase {
    constructor({
        name, frameRate, dropFrame = false, startTime = 0.0,
        startTimecode = "00:00:00:00", startFrame = 0, notes = []
    }) {
        this.name = name || "Main Timeline";
        this.frameRate = frameRate;
        this.dropFrame = dropFrame;
        this.startTime = startTime;
        this.startTimecode = startTimecode;
        this.startFrame = startFrame;
        this.notes = notes; // Array of Note objects
    }
}

export class Sequence extends SequenceFileBase {
    constructor(params = {}) {
        super(params);
    }
}

export class File extends SequenceFileBase {
    constructor({ fileName = "", filePath = "", clipName, ...baseParams }) {
        super(baseParams);
        this.fileName = fileName; // Required
        this.filePath = filePath; // Required
        this.clipName = clipName;
    }
}

export class OTRNDocument {
    constructor({ metadata, sequence, files = [] } = {}) {
        this.metadata = metadata || new Metadata();
        this.sequence = sequence || new Sequence();
        this.files = files; // Array of File objects
    }

    /**
     * Converts the OTRNDocument and all nested classes to a clean dictionary/object.
     * @returns {Object}
     */
    toDict() {
        return cleanToDict(this);
    }
}