import json
import uuid
from datetime import datetime
from dataclasses import dataclass, field, asdict
from typing import List, Dict, Any, Optional

@dataclass
class Reply:
    """Represents a single reply object within a Note (Section 8)."""
    comment: str  # Required
    commenter: Optional[str] = None
    timestamp: Optional[str] = None  # ISO 8601
    replyId: Optional[str] = None

@dataclass
class Tag:
    """Represents a single tag object within a Note (Section 9)."""
    name: str  # Required
    tagId: Optional[str] = None
    group: Optional[str] = None

@dataclass
class Note:
    """Represents a single timecode-related note, including range and nested objects (Section 7)."""
    # Required Attributes
    time: float  # The exact time in seconds
    comment: str # The main body of the note
    
    # Optional Timecode Attributes
    timecode: Optional[str] = None  # HH:MM:SS:FF or HH:MM:SS;FF
    frame: Optional[int] = None
    
    # Optional Range Marker Attributes
    range: bool = False
    timeOut: Optional[float] = None
    timecodeOut: Optional[str] = None
    frameOut: Optional[int] = None
    
    # Optional Note Metadata Attributes (CamelCase)
    name: Optional[str] = None
    commenter: Optional[str] = None
    color: Optional[str] = None          # Color name (e.g., "red")
    colorHex: Optional[str] = None       # Hex color code (e.g., "#FF0000")
    category: Optional[str] = None
    track: Optional[str] = None          # Track (e.g., "V1", "A2")
    complete: bool = False               # Status flag
    timestamp: Optional[str] = None      # ISO 8601 creation/update time
    noteId: Optional[str] = field(default_factory=lambda: str(uuid.uuid4())) # Unique ID
    
    # Optional Nested Arrays
    replies: List[Reply] = field(default_factory=list)
    tags: List[Tag] = field(default_factory=list)

@dataclass
class Metadata:
    """Represents the top-level 'metadata' object using CamelCase (Section 5)."""
    project: Optional[str] = None
    fileName: Optional[str] = None
    fileUrl: Optional[str] = None
    software: Optional[str] = None
    otrnVersion: int = 1  # Default version 1
    otrnInfo: Optional[str] = "This is an OTRN (Open Timecode-Related Notes) notes file. Learn more about the specification on https://otrn.editingtools.io"
    timestamp: Optional[str] = field(default_factory=lambda: datetime.now().isoformat(timespec='seconds') + 'Z') # ISO 8601 with Z

@dataclass
class SequenceFileBase:
    """Base class for Sequence and File objects, holding common timecode attributes (Sections 6 & 10)."""
    # Optional (recommended) Attributes
    name: Optional[str] = None
    frameRate: Optional[float] = None
    dropFrame: bool = False
    
    # Optional Time Attributes
    startTime: float = 0.0
    startTimecode: str = "00:00:00:00"
    startFrame: int = 0
    
    # Required array
    notes: List[Note] = field(default_factory=list)

@dataclass
class Sequence(SequenceFileBase):
    """Represents the top-level 'sequence' object (Section 6)."""
    pass # Inherits all fields from SequenceFileBase

@dataclass
class File(SequenceFileBase):
    """Represents a single object in the top-level 'files' array for clip notes (Section 10)."""
    fileName: str = ""  # Required
    filePath: str = ""  # Required
    clipName: Optional[str] = None

@dataclass
class OTRNDocument:
    """
    The root structure for an Open Timecode-Related Notes file (Section 4).
    Contains metadata, sequence (required), and files (optional).
    """
    metadata: Metadata = field(default_factory=Metadata)
    sequence: Sequence = field(default_factory=Sequence)
    files: List[File] = field(default_factory=list) # Optional array
    
    def to_dict(self) -> Dict[str, Any]:
        """
        Converts the OTRNDocument and all nested dataclasses to a dictionary.
        Cleans up None values but keeps required empty arrays (like notes, replies, tags).
        """
        # Use asdict to recursively convert the dataclass to a dictionary
        data = asdict(self)
        
        # Clean up None values globally
        def clean_none_values(d):
            if isinstance(d, dict):
                # Only keep keys with None value if they represent required arrays
                required_empty_lists = ['notes', 'replies', 'tags'] 
                return {k: clean_none_values(v) for k, v in d.items() if v is not None or k in required_empty_lists}
            elif isinstance(d, list):
                return [clean_none_values(v) for v in d]
            return d
            
        cleaned_data = clean_none_values(data)
        
        # Remove the 'files' array if it is empty, as it is optional at the root
        if 'files' in cleaned_data and not cleaned_data['files']:
             del cleaned_data['files']
             
        return cleaned_data

# --- Library Functions ---
def _instantiate_list(cls, data_list: List[Dict]) -> List[Any]:
    """Helper to instantiate a list of dataclasses from a raw list of dictionaries."""
    return [cls(**item) for item in data_list]

def read_otrn(filepath: str) -> OTRNDocument:
    """Reads an OTRN JSON file and returns an OTRNDocument object."""
    print(f"Reading OTRN file from: {filepath}")
    with open(filepath, 'r', encoding='utf-8') as f:
        data = json.load(f)
    
    # Helper function to process notes, including nested replies and tags
    def process_notes(notes_data: List[Dict]) -> List[Note]:
        notes = []
        for note_data in notes_data:
            # Instantiate nested arrays first, removing them from the dictionary
            note_data['replies'] = _instantiate_list(Reply, note_data.pop('replies', []))
            note_data['tags'] = _instantiate_list(Tag, note_data.pop('tags', []))
            notes.append(Note(**note_data))
        return notes

    # 1. Process Metadata
    metadata = Metadata(**data.get('metadata', {}))

    # 2. Process Sequence
    sequence_data = data.get('sequence', {})
    sequence_data['notes'] = process_notes(sequence_data.get('notes', []))
    sequence = Sequence(**sequence_data)
    
    # 3. Process Files (optional)
    files = []
    files_data = data.get('files', [])
    for file_data in files_data:
        file_data['notes'] = process_notes(file_data.get('notes', []))
        files.append(File(**file_data))

    # 4. Construct the OTRNDocument
    document = OTRNDocument(
        metadata=metadata,
        sequence=sequence,
        files=files
    )
    
    print("Successfully loaded OTRN document.")
    return document

def write_otrn(document: OTRNDocument, filepath: str):
    """Writes an OTRNDocument object to a JSON file."""
    print(f"Writing OTRN document to: {filepath}")
    
    # Convert the document object to a clean dictionary
    data = document.to_dict()

    with open(filepath, 'w', encoding='utf-8') as f:
        # Use indent=4 for human-readable output
        json.dump(data, f, indent=4)
    
    print("Successfully wrote OTRN document.")
