from otrn_lib import OTRNDocument, Metadata, Sequence, Note, File, Reply, Tag, write_otrn
from datetime import datetime

# Define the file name that the writer will output
OTRN_OUTPUT_FILE = "example_data_to_read.otrn"

def create_example_document() -> OTRNDocument:
    """Creates a fully populated OTRNDocument object ready for serialization."""
    
    # Define a note
    note1 = Note(
        time=300.0,
        comment="Great visual transition here. Flag for the sizzle reel.",
        commenter="Director",
        color="lime",
        complete=True,
        noteId="seq-n1"
    )
    
    # Define a note with replies
    # 1a. Define Metadata
    metadata_obj = Metadata(
        project="Quantum Mechanics Documentary (Writer)",
        fileName="Episode_Pilot_V03.mov",
        fileUrl="ftp://server.com/docs/Episode_Pilot_V03.mov",
        software="OTRN Python Writer",
        otrnVersion=1
    )

    # 1b. Define nested Reply and Tag
    reply_obj = Reply(
        commenter="Sound Mixer",
        comment="This audio issue is fixed in the latest mix (V05).",
        timestamp=datetime.now().isoformat(timespec='seconds') + 'Z',
        replyId="rep-12345"
    )
    
    tag_obj = Tag(
        name="Archive",
        tagId="archive_tag",
        group="Footage Management"
    )

    # 1c. Define Notes for the Sequence
    note2 = Note(
        time=120.5,  # Required: Time in seconds
        timecode="01:02:00:12",
        comment="The host's microphone pops here. Needs de-essing or replacement.", # Required
        commenter="Audio Editor",
        color="orange",
        category="Audio",
        track="A1",
        complete=False,
        range=True,
        timeOut=122.8,
        timecodeOut="01:02:02:07",
        noteId="seq-n2",
        replies=[reply_obj],
        tags=[tag_obj]
    )

    # Create the Sequence object
    sequence_obj = Sequence(
        name="Final Edit Timeline",
        frameRate=24.0,
        dropFrame=False,
        startTimecode="01:00:00:00",
        notes=[note1, note2]
    )
    
    
    
    # Optional, Create notes for files (file marker)
    # 2. Create a File object (for clip notes)
    file_note = Note(
        time=5.0,
        timecode="14:24:15:00",
        comment="The interviewer is visibly shifting in their seat. Trim this for the final cut.",
        commenter="Transcriptionist"
    )
    file_obj = File(
        fileName="INTERVIEW_S01_T04.mov",
        filePath="/path/to/rushes/INTERVIEW_S01_T04.mov",
        clipName="Interview Subject A - Take 4",
        frameRate=25.0,
        startTimecode="14:24:10:00",
        notes=[file_note]
    )



    # 3. Create the full OTRNDocument
    new_doc = OTRNDocument(
        metadata=metadata_obj,
        sequence=sequence_obj,
        files=[file_obj]
    )
    
    return new_doc

def main_write_example():

    print("--- Starting OTRN Writer Example ---")
    new_doc = create_example_document()
    
    print(f"Document created for project: {new_doc.metadata.project}")
    print(f"Writing document to: {OTRN_OUTPUT_FILE}")

    # Call the write function from the library
    write_otrn(new_doc, OTRN_OUTPUT_FILE)
    
    print(f"\nWriter example finished. A file named '{OTRN_OUTPUT_FILE}' has been created.")

if __name__ == "__main__":
    main_write_example()
